/**
 * Fashion Boutique Theme JavaScript
 */

(function($) {
    'use strict';

    // DOM Ready
    $(document).ready(function() {
        // Initialize all functions
        initMobileMenu();
        initSmoothScrolling();
        initNewsletterForm();
        initProductInteractions();
        initSearchFunctionality();
        initAnimations();
    });

    /**
     * Mobile Menu Toggle
     */
    function initMobileMenu() {
        $('.mobile-toggle').on('click', function() {
            $(this).toggleClass('active');
            $('.nav-menu').toggleClass('active');
            
            // Animate hamburger icon
            $(this).find('span').toggleClass('active');
        });

        // Close mobile menu when clicking outside
        $(document).on('click', function(e) {
            if (!$(e.target).closest('.navbar').length) {
                $('.mobile-toggle').removeClass('active');
                $('.nav-menu').removeClass('active');
            }
        });

        // Close mobile menu when window is resized
        $(window).on('resize', function() {
            if ($(window).width() > 768) {
                $('.mobile-toggle').removeClass('active');
                $('.nav-menu').removeClass('active');
            }
        });
    }

    /**
     * Smooth Scrolling for Anchor Links
     */
    function initSmoothScrolling() {
        $('a[href*="#"]:not([href="#"])').on('click', function() {
            if (location.pathname.replace(/^\//, '') == this.pathname.replace(/^\//, '') && location.hostname == this.hostname) {
                var target = $(this.hash);
                target = target.length ? target : $('[name=' + this.hash.slice(1) + ']');
                if (target.length) {
                    $('html, body').animate({
                        scrollTop: target.offset().top - 100
                    }, 1000);
                    return false;
                }
            }
        });
    }

    /**
     * Newsletter Form Submission
     */
    function initNewsletterForm() {
        $('#newsletter-form').on('submit', function(e) {
            e.preventDefault();
            
            var email = $(this).find('input[type="email"]').val();
            var $message = $('#newsletter-message');
            var $button = $(this).find('button[type="submit"]');
            
            // Basic email validation
            if (!isValidEmail(email)) {
                showMessage('Please enter a valid email address.', 'error');
                return;
            }

            // Show loading state
            $button.text('Subscribing...').prop('disabled', true);

            // AJAX request
            $.ajax({
                url: fashion_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'newsletter_signup',
                    email: email,
                    nonce: fashion_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        showMessage(response.data, 'success');
                        $('#newsletter-form')[0].reset();
                    } else {
                        showMessage(response.data, 'error');
                    }
                },
                error: function() {
                    showMessage('Something went wrong. Please try again.', 'error');
                },
                complete: function() {
                    $button.text('Subscribe').prop('disabled', false);
                }
            });
        });

        function showMessage(message, type) {
            var $message = $('#newsletter-message');
            $message.text(message)
                   .removeClass('success error')
                   .addClass(type)
                   .show()
                   .delay(5000)
                   .fadeOut();
        }

        function isValidEmail(email) {
            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }
    }

    /**
     * Product Interactions
     */
    function initProductInteractions() {
        // Add to Cart functionality
        $('.add-to-cart-btn, .quick-add-btn').on('click', function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var productId = $button.data('product-id');
            var originalText = $button.text();
            
            // Show loading state
            $button.text('Adding...').prop('disabled', true);
            
            // Simulate add to cart (replace with actual WooCommerce AJAX)
            setTimeout(function() {
                $button.text('Added!').removeClass('add-to-cart-btn').addClass('added-to-cart');
                
                // Update cart count if WooCommerce is active
                updateCartCount();
                
                // Reset button after 2 seconds
                setTimeout(function() {
                    $button.text(originalText).prop('disabled', false);
                    $button.removeClass('added-to-cart').addClass('add-to-cart-btn');
                }, 2000);
            }, 500);
        });

        // Product card hover effects
        $('.product-card').on('mouseenter', function() {
            $(this).find('.product-overlay').fadeIn(300);
        }).on('mouseleave', function() {
            $(this).find('.product-overlay').fadeOut(300);
        });

        // Category card interactions
        $('.category-card').on('click', function(e) {
            e.preventDefault();
            var categoryName = $(this).find('.category-title').text();
            
            // Add loading effect
            $(this).addClass('loading');
            
            // Simulate navigation (replace with actual links)
            setTimeout(function() {
                // window.location.href = $(this).attr('href');
                console.log('Navigate to ' + categoryName + ' category');
            }, 500);
        });
    }

    /**
     * Search Functionality
     */
    function initSearchFunctionality() {
        var $searchInput = $('.search-input');
        var $searchForm = $('.search-form');
        
        // Search suggestions (if you want to implement)
        $searchInput.on('input', function() {
            var query = $(this).val();
            if (query.length > 2) {
                // Implement search suggestions here
                console.log('Search for: ' + query);
            }
        });

        // Search form submission
        $searchForm.on('submit', function(e) {
            var query = $searchInput.val().trim();
            if (query === '') {
                e.preventDefault();
                $searchInput.focus();
            }
        });
    }

    /**
     * Scroll Animations
     */
    function initAnimations() {
        // Fade in elements on scroll
        function fadeInOnScroll() {
            $('.fade-in').each(function() {
                var elementTop = $(this).offset().top;
                var windowTop = $(window).scrollTop();
                var windowHeight = $(window).height();
                
                if (elementTop < windowTop + windowHeight - 100) {
                    $(this).addClass('animated');
                }
            });
        }
        
        // Initial check
        fadeInOnScroll();
        
        // Check on scroll
        $(window).on('scroll', throttle(fadeInOnScroll, 100));
        
        // Header scroll effect
        $(window).on('scroll', throttle(function() {
            var scrollTop = $(window).scrollTop();
            if (scrollTop > 100) {
                $('.site-header').addClass('scrolled');
            } else {
                $('.site-header').removeClass('scrolled');
            }
        }, 100));
    }

    /**
     * Update Cart Count
     */
    function updateCartCount() {
        if (typeof fashion_ajax !== 'undefined') {
            $.ajax({
                url: fashion_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'update_cart_count',
                    nonce: fashion_ajax.nonce
                },
                success: function(response) {
                    if (response.count > 0) {
                        $('#cart-count').text(response.count).show();
                    } else {
                        $('#cart-count').hide();
                    }
                }
            });
        }
    }

    /**
     * Throttle function for performance
     */
    function throttle(func, wait) {
        var context, args, result;
        var timeout = null;
        var previous = 0;
        
        var later = function() {
            previous = Date.now();
            timeout = null;
            result = func.apply(context, args);
            if (!timeout) context = args = null;
        };
        
        return function() {
            var now = Date.now();
            var remaining = wait - (now - previous);
            context = this;
            args = arguments;
            
            if (remaining <= 0 || remaining > wait) {
                if (timeout) {
                    clearTimeout(timeout);
                    timeout = null;
                }
                previous = now;
                result = func.apply(context, args);
                if (!timeout) context = args = null;
            } else if (!timeout) {
                timeout = setTimeout(later, remaining);
            }
            return result;
        };
    }

    /**
     * Lazy Loading for Images (if needed)
     */
    function initLazyLoading() {
        if ('IntersectionObserver' in window) {
            var imageObserver = new IntersectionObserver(function(entries, observer) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        var img = entry.target;
                        img.src = img.dataset.src;
                        img.classList.remove('lazy');
                        imageObserver.unobserve(img);
                    }
                });
            });

            document.querySelectorAll('img[data-src]').forEach(function(img) {
                imageObserver.observe(img);
            });
        }
    }

})(jQuery);