<?php
/**
 * Fashion Boutique Theme Functions
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Theme Setup
 */
function fashion_boutique_setup() {
    // Add theme support for various features
    add_theme_support('title-tag');
    add_theme_support('post-thumbnails');
    add_theme_support('custom-logo');
    add_theme_support('html5', array('search-form', 'comment-form', 'comment-list', 'gallery', 'caption'));
    add_theme_support('customize-selective-refresh-widgets');
    
    // Add WooCommerce support
    add_theme_support('woocommerce');
    add_theme_support('wc-product-gallery-zoom');
    add_theme_support('wc-product-gallery-lightbox');
    add_theme_support('wc-product-gallery-slider');
    
    // Register navigation menus
    register_nav_menus(array(
        'primary' => __('Primary Menu', 'fashion-boutique'),
        'footer' => __('Footer Menu', 'fashion-boutique'),
    ));
}
add_action('after_setup_theme', 'fashion_boutique_setup');

/**
 * Enqueue Scripts and Styles
 */
function fashion_boutique_scripts() {
    // Enqueue styles
    wp_enqueue_style('fashion-boutique-style', get_stylesheet_uri(), array(), '1.0.0');
    wp_enqueue_style('font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css', array(), '6.4.0');
    wp_enqueue_style('google-fonts', 'https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap', array(), null);
    
    // Enqueue scripts
    wp_enqueue_script('fashion-boutique-main', get_template_directory_uri() . '/js/main.js', array('jquery'), '1.0.0', true);
    
    // Localize script for AJAX
    wp_localize_script('fashion-boutique-main', 'fashion_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('fashion_nonce')
    ));
}
add_action('wp_enqueue_scripts', 'fashion_boutique_scripts');

/**
 * Register Widget Areas
 */
function fashion_boutique_widgets_init() {
    register_sidebar(array(
        'name'          => __('Sidebar', 'fashion-boutique'),
        'id'            => 'sidebar-1',
        'description'   => __('Add widgets here.', 'fashion-boutique'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ));
    
    register_sidebar(array(
        'name'          => __('Footer 1', 'fashion-boutique'),
        'id'            => 'footer-1',
        'description'   => __('Footer widget area 1', 'fashion-boutique'),
        'before_widget' => '<div class="footer-widget">',
        'after_widget'  => '</div>',
        'before_title'  => '<h4>',
        'after_title'   => '</h4>',
    ));
    
    register_sidebar(array(
        'name'          => __('Footer 2', 'fashion-boutique'),
        'id'            => 'footer-2',
        'description'   => __('Footer widget area 2', 'fashion-boutique'),
        'before_widget' => '<div class="footer-widget">',
        'after_widget'  => '</div>',
        'before_title'  => '<h4>',
        'after_title'   => '</h4>',
    ));
    
    register_sidebar(array(
        'name'          => __('Footer 3', 'fashion-boutique'),
        'id'            => 'footer-3',
        'description'   => __('Footer widget area 3', 'fashion-boutique'),
        'before_widget' => '<div class="footer-widget">',
        'after_widget'  => '</div>',
        'before_title'  => '<h4>',
        'after_title'   => '</h4>',
    ));
}
add_action('widgets_init', 'fashion_boutique_widgets_init');

/**
 * Custom Logo Setup
 */
function fashion_boutique_custom_logo_setup() {
    $defaults = array(
        'height'      => 100,
        'width'       => 400,
        'flex-height' => true,
        'flex-width'  => true,
        'header-text' => array('site-title', 'site-description'),
    );
    add_theme_support('custom-logo', $defaults);
}
add_action('after_setup_theme', 'fashion_boutique_custom_logo_setup');

/**
 * Customizer Settings
 */
function fashion_boutique_customize_register($wp_customize) {
    // Hero Section
    $wp_customize->add_section('hero_section', array(
        'title'    => __('Hero Section', 'fashion-boutique'),
        'priority' => 30,
    ));
    
    $wp_customize->add_setting('hero_title', array(
        'default'           => 'Elegant Fashion for Modern Women',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('hero_title', array(
        'label'    => __('Hero Title', 'fashion-boutique'),
        'section'  => 'hero_section',
        'type'     => 'text',
    ));
    
    $wp_customize->add_setting('hero_subtitle', array(
        'default'           => 'Discover our curated collection of timeless pieces',
        'sanitize_callback' => 'sanitize_textarea_field',
    ));
    
    $wp_customize->add_control('hero_subtitle', array(
        'label'    => __('Hero Subtitle', 'fashion-boutique'),
        'section'  => 'hero_section',
        'type'     => 'textarea',
    ));
    
    $wp_customize->add_setting('hero_button_text', array(
        'default'           => 'Shop Now',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('hero_button_text', array(
        'label'    => __('Hero Button Text', 'fashion-boutique'),
        'section'  => 'hero_section',
        'type'     => 'text',
    ));
    
    $wp_customize->add_setting('hero_button_link', array(
        'default'           => '#',
        'sanitize_callback' => 'esc_url_raw',
    ));
    
    $wp_customize->add_control('hero_button_link', array(
        'label'    => __('Hero Button Link', 'fashion-boutique'),
        'section'  => 'hero_section',
        'type'     => 'url',
    ));
    
    // Promotional Banner
    $wp_customize->add_section('promo_section', array(
        'title'    => __('Promotional Banner', 'fashion-boutique'),
        'priority' => 35,
    ));
    
    $wp_customize->add_setting('promo_title', array(
        'default'           => 'SUMMER SALE',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('promo_title', array(
        'label'    => __('Promo Title', 'fashion-boutique'),
        'section'  => 'promo_section',
        'type'     => 'text',
    ));
    
    $wp_customize->add_setting('promo_subtitle', array(
        'default'           => 'Up to 50% off selected items',
        'sanitize_callback' => 'sanitize_textarea_field',
    ));
    
    $wp_customize->add_control('promo_subtitle', array(
        'label'    => __('Promo Subtitle', 'fashion-boutique'),
        'section'  => 'promo_section',
        'type'     => 'textarea',
    ));
    
    // Social Media
    $wp_customize->add_section('social_section', array(
        'title'    => __('Social Media', 'fashion-boutique'),
        'priority' => 40,
    ));
    
    $social_networks = array(
        'facebook' => 'Facebook',
        'instagram' => 'Instagram',
        'twitter' => 'Twitter',
        'pinterest' => 'Pinterest'
    );
    
    foreach ($social_networks as $network => $label) {
        $wp_customize->add_setting("social_{$network}", array(
            'default'           => '',
            'sanitize_callback' => 'esc_url_raw',
        ));
        
        $wp_customize->add_control("social_{$network}", array(
            'label'    => sprintf(__('%s URL', 'fashion-boutique'), $label),
            'section'  => 'social_section',
            'type'     => 'url',
        ));
    }
}
add_action('customize_register', 'fashion_boutique_customize_register');

/**
 * Get Featured Products (for WooCommerce)
 */
function get_featured_products($limit = 4) {
    if (!class_exists('WooCommerce')) {
        return array();
    }
    
    $args = array(
        'post_type' => 'product',
        'posts_per_page' => $limit,
        'meta_query' => array(
            array(
                'key' => '_featured',
                'value' => 'yes'
            )
        )
    );
    
    return get_posts($args);
}

/**
 * Get Product Categories
 */
function get_product_categories($limit = 3) {
    if (!class_exists('WooCommerce')) {
        return array();
    }
    
    return get_terms(array(
        'taxonomy' => 'product_cat',
        'number' => $limit,
        'hide_empty' => false
    ));
}

/**
 * AJAX Newsletter Signup
 */
function fashion_boutique_newsletter_signup() {
    check_ajax_referer('fashion_nonce', 'nonce');
    
    $email = sanitize_email($_POST['email']);
    
    if (!is_email($email)) {
        wp_die('Invalid email address');
    }
    
    // Here you can integrate with your newsletter service
    // For now, we'll just save to WordPress options or a custom table
    $subscribers = get_option('fashion_newsletter_subscribers', array());
    
    if (!in_array($email, $subscribers)) {
        $subscribers[] = $email;
        update_option('fashion_newsletter_subscribers', $subscribers);
        wp_send_json_success('Thank you for subscribing!');
    } else {
        wp_send_json_error('Email already subscribed');
    }
}
add_action('wp_ajax_newsletter_signup', 'fashion_boutique_newsletter_signup');
add_action('wp_ajax_nopriv_newsletter_signup', 'fashion_boutique_newsletter_signup');

/**
 * Custom Post Type for Testimonials (Optional)
 */
function fashion_boutique_custom_post_types() {
    register_post_type('testimonial', array(
        'labels' => array(
            'name' => 'Testimonials',
            'singular_name' => 'Testimonial'
        ),
        'public' => true,
        'supports' => array('title', 'editor', 'thumbnail'),
        'menu_icon' => 'dashicons-format-quote'
    ));
}
add_action('init', 'fashion_boutique_custom_post_types');

/**
 * Add Cart Count to Header
 */
function get_cart_count() {
    if (class_exists('WooCommerce')) {
        return WC()->cart->get_cart_contents_count();
    }
    return 0;
}

/**
 * Update cart count via AJAX
 */
function update_cart_count() {
    wp_send_json(array('count' => get_cart_count()));
}
add_action('wp_ajax_update_cart_count', 'update_cart_count');
add_action('wp_ajax_nopriv_update_cart_count', 'update_cart_count');

/**
 * Custom Excerpt Length
 */
function fashion_boutique_excerpt_length($length) {
    return 20;
}
add_filter('excerpt_length', 'fashion_boutique_excerpt_length');

/**
 * Custom Excerpt More
 */
function fashion_boutique_excerpt_more($more) {
    return '...';
}
add_filter('excerpt_more', 'fashion_boutique_excerpt_more');

/**
 * Create default pages on theme activation
 */
function fashion_boutique_create_default_pages() {
    // Check if pages already exist to avoid duplicates
    $existing_pages = get_posts(array(
        'post_type' => 'page',
        'meta_key' => '_fashion_boutique_default_page',
        'meta_value' => 'yes',
        'posts_per_page' => -1
    ));
    
    if (!empty($existing_pages)) {
        return; // Pages already created
    }
    
    // Pages to create
    $default_pages = array(
        'Home' => array(
            'title' => 'Home',
            'content' => '<!-- This is the homepage content. The theme will display the fashion boutique layout automatically. -->
            <p>Welcome to our fashion boutique! This page showcases our featured products, categories, and promotional content.</p>',
            'template' => 'page-home.php'
        ),
        'About Us' => array(
            'title' => 'About Us',
            'content' => '<h2>Our Story</h2>
            <p>Welcome to Fashion Boutique, where style meets sophistication. We are passionate about bringing you the finest selection of contemporary fashion pieces that embody both elegance and modern flair.</p>
            
            <h3>Our Mission</h3>
            <p>To provide fashion-forward individuals with carefully curated clothing and accessories that enhance their personal style and confidence.</p>
            
            <h3>What Makes Us Different</h3>
            <ul>
            <li>Handpicked selection of high-quality pieces</li>
            <li>Focus on timeless elegance with contemporary trends</li>
            <li>Exceptional customer service and styling advice</li>
            <li>Sustainable and ethical fashion choices</li>
            </ul>',
            'template' => ''
        ),
        'Contact' => array(
            'title' => 'Contact Us',
            'content' => '<h2>Get in Touch</h2>
            <p>We would love to hear from you! Contact us for any questions about our products, styling advice, or general inquiries.</p>
            
            <div class="contact-info">
            <h3>Store Information</h3>
            <p><strong>Address:</strong> 123 Fashion Street, Style City, SC 12345</p>
            <p><strong>Phone:</strong> +1 (555) 123-4567</p>
            <p><strong>Email:</strong> hello@fashionboutique.com</p>
            
            <h3>Store Hours</h3>
            <p><strong>Monday - Saturday:</strong> 10:00 AM - 7:00 PM</p>
            <p><strong>Sunday:</strong> 12:00 PM - 6:00 PM</p>
            </div>',
            'template' => ''
        ),
        'Privacy Policy' => array(
            'title' => 'Privacy Policy',
            'content' => '<h2>Privacy Policy</h2>
            <p>Your privacy is important to us. This privacy policy explains how we collect, use, and protect your personal information.</p>
            
            <h3>Information We Collect</h3>
            <p>We may collect personal information such as your name, email address, phone number, and shipping address when you make a purchase or sign up for our newsletter.</p>
            
            <h3>How We Use Your Information</h3>
            <ul>
            <li>To process and fulfill your orders</li>
            <li>To send you important updates about your orders</li>
            <li>To improve our products and services</li>
            <li>To send you promotional emails (with your consent)</li>
            </ul>
            
            <h3>Data Protection</h3>
            <p>We implement appropriate security measures to protect your personal information against unauthorized access, alteration, disclosure, or destruction.</p>',
            'template' => ''
        ),
        'Shipping & Returns' => array(
            'title' => 'Shipping & Returns',
            'content' => '<h2>Shipping Information</h2>
            
            <h3>Shipping Options</h3>
            <ul>
            <li><strong>Standard Shipping:</strong> 5-7 business days - Free on orders over $100</li>
            <li><strong>Express Shipping:</strong> 2-3 business days - $15</li>
            <li><strong>Next Day Delivery:</strong> 1 business day - $25</li>
            </ul>
            
            <h2>Returns Policy</h2>
            
            <h3>30-Day Return Policy</h3>
            <p>We offer a 30-day return policy on most items. Items must be in original condition with tags attached.</p>
            
            <h3>How to Return</h3>
            <ol>
            <li>Contact our customer service team</li>
            <li>Receive your return authorization and shipping label</li>
            <li>Pack items securely and ship back to us</li>
            <li>Refund will be processed within 5-7 business days</li>
            </ol>
            
            <h3>Exchange Policy</h3>
            <p>We gladly accept exchanges for different sizes or colors within 30 days of purchase.</p>',
            'template' => ''
        ),
        'Size Guide' => array(
            'title' => 'Size Guide',
            'content' => '<h2>Size Guide</h2>
            <p>Find your perfect fit with our comprehensive size guide. All measurements are in inches.</p>
            
            <h3>Women\'s Clothing</h3>
            <table style="border-collapse: collapse; width: 100%; margin: 20px 0;">
            <tr style="background-color: #f8f8f8;">
            <th style="border: 1px solid #ddd; padding: 12px;">Size</th>
            <th style="border: 1px solid #ddd; padding: 12px;">Bust</th>
            <th style="border: 1px solid #ddd; padding: 12px;">Waist</th>
            <th style="border: 1px solid #ddd; padding: 12px;">Hips</th>
            </tr>
            <tr><td style="border: 1px solid #ddd; padding: 8px;">XS</td><td style="border: 1px solid #ddd; padding: 8px;">32-33</td><td style="border: 1px solid #ddd; padding: 8px;">24-25</td><td style="border: 1px solid #ddd; padding: 8px;">34-35</td></tr>
            <tr><td style="border: 1px solid #ddd; padding: 8px;">S</td><td style="border: 1px solid #ddd; padding: 8px;">34-35</td><td style="border: 1px solid #ddd; padding: 8px;">26-27</td><td style="border: 1px solid #ddd; padding: 8px;">36-37</td></tr>
            <tr><td style="border: 1px solid #ddd; padding: 8px;">M</td><td style="border: 1px solid #ddd; padding: 8px;">36-37</td><td style="border: 1px solid #ddd; padding: 8px;">28-29</td><td style="border: 1px solid #ddd; padding: 8px;">38-39</td></tr>
            <tr><td style="border: 1px solid #ddd; padding: 8px;">L</td><td style="border: 1px solid #ddd; padding: 8px;">38-40</td><td style="border: 1px solid #ddd; padding: 8px;">30-32</td><td style="border: 1px solid #ddd; padding: 8px;">40-42</td></tr>
            <tr><td style="border: 1px solid #ddd; padding: 8px;">XL</td><td style="border: 1px solid #ddd; padding: 8px;">41-43</td><td style="border: 1px solid #ddd; padding: 8px;">33-35</td><td style="border: 1px solid #ddd; padding: 8px;">43-45</td></tr>
            </table>
            
            <h3>How to Measure</h3>
            <ul>
            <li><strong>Bust:</strong> Measure around the fullest part of your bust</li>
            <li><strong>Waist:</strong> Measure around the narrowest part of your waist</li>
            <li><strong>Hips:</strong> Measure around the fullest part of your hips</li>
            </ul>',
            'template' => ''
        )
    );
    
    // Create pages
    foreach ($default_pages as $page_slug => $page_data) {
        $page_id = wp_insert_post(array(
            'post_title' => $page_data['title'],
            'post_content' => $page_data['content'],
            'post_status' => 'publish',
            'post_type' => 'page',
            'post_name' => sanitize_title($page_data['title'])
        ));
        
        if ($page_id && !is_wp_error($page_id)) {
            // Mark as default theme page
            update_post_meta($page_id, '_fashion_boutique_default_page', 'yes');
            
            // Set page template if specified
            if (!empty($page_data['template'])) {
                update_post_meta($page_id, '_wp_page_template', $page_data['template']);
            }
            
            // Set Home page as front page
            if ($page_data['title'] === 'Home') {
                update_option('show_on_front', 'page');
                update_option('page_on_front', $page_id);
            }
        }
    }
    
    // Create a menu with the new pages
    fashion_boutique_create_default_menu();
}

/**
 * Create default navigation menu
 */
function fashion_boutique_create_default_menu() {
    // Check if menu already exists
    $menu_name = 'Fashion Boutique Menu';
    $menu_exists = wp_get_nav_menu_object($menu_name);
    
    if ($menu_exists) {
        return; // Menu already exists
    }
    
    // Create the menu
    $menu_id = wp_create_nav_menu($menu_name);
    
    if (is_wp_error($menu_id)) {
        return;
    }
    
    // Get pages to add to menu
    $pages = get_pages(array(
        'meta_key' => '_fashion_boutique_default_page',
        'meta_value' => 'yes',
        'sort_column' => 'post_title'
    ));
    
    $menu_order = 1;
    
    // Add Home page first
    foreach ($pages as $page) {
        if ($page->post_title === 'Home') {
            wp_update_nav_menu_item($menu_id, 0, array(
                'menu-item-title' => 'Home',
                'menu-item-object' => 'page',
                'menu-item-object-id' => $page->ID,
                'menu-item-type' => 'post_type',
                'menu-item-status' => 'publish',
                'menu-item-position' => $menu_order++
            ));
            break;
        }
    }
    
    // Add WooCommerce pages if WooCommerce is active
    if (class_exists('WooCommerce')) {
        // Shop page
        $shop_page_id = wc_get_page_id('shop');
        if ($shop_page_id > 0) {
            wp_update_nav_menu_item($menu_id, 0, array(
                'menu-item-title' => 'Shop',
                'menu-item-object' => 'page',
                'menu-item-object-id' => $shop_page_id,
                'menu-item-type' => 'post_type',
                'menu-item-status' => 'publish',
                'menu-item-position' => $menu_order++
            ));
        }
        
        // Add product categories
        $categories = get_terms(array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
            'number' => 3
        ));
        
        foreach ($categories as $category) {
            wp_update_nav_menu_item($menu_id, 0, array(
                'menu-item-title' => $category->name,
                'menu-item-object' => 'product_cat',
                'menu-item-object-id' => $category->term_id,
                'menu-item-type' => 'taxonomy',
                'menu-item-status' => 'publish',
                'menu-item-position' => $menu_order++
            ));
        }
    }
    
    // Add other pages
    $page_order = array('About Us', 'Contact', 'Size Guide');
    foreach ($page_order as $page_title) {
        foreach ($pages as $page) {
            if ($page->post_title === $page_title) {
                wp_update_nav_menu_item($menu_id, 0, array(
                    'menu-item-title' => $page->post_title,
                    'menu-item-object' => 'page',
                    'menu-item-object-id' => $page->ID,
                    'menu-item-type' => 'post_type',
                    'menu-item-status' => 'publish',
                    'menu-item-position' => $menu_order++
                ));
                break;
            }
        }
    }
    
    // Assign menu to theme location
    $locations = get_theme_mod('nav_menu_locations');
    $locations['primary'] = $menu_id;
    set_theme_mod('nav_menu_locations', $locations);
}

/**
 * Run setup on theme activation
 */
function fashion_boutique_activation_setup() {
    fashion_boutique_create_default_pages();
    
    // Flush rewrite rules
    flush_rewrite_rules();
    
    // Set permalink structure
    global $wp_rewrite;
    $wp_rewrite->set_permalink_structure('/%postname%/');
    $wp_rewrite->flush_rules();
}
add_action('after_switch_theme', 'fashion_boutique_activation_setup');

/**
 * Also run setup on admin init to ensure pages exist
 */
function fashion_boutique_admin_init_setup() {
    // Only run if we're in admin and no default pages exist
    if (is_admin()) {
        $existing_pages = get_posts(array(
            'post_type' => 'page',
            'meta_key' => '_fashion_boutique_default_page',
            'meta_value' => 'yes',
            'posts_per_page' => 1
        ));
        
        if (empty($existing_pages)) {
            fashion_boutique_create_default_pages();
        }
    }
}
add_action('admin_init', 'fashion_boutique_admin_init_setup');

/**
 * Add admin notice to guide users
 */
function fashion_boutique_admin_notice() {
    global $pagenow;
    
    if ($pagenow == 'themes.php' && isset($_GET['activated'])) {
        echo '<div class="notice notice-success is-dismissible">';
        echo '<p><strong>Fashion Boutique Theme Activated!</strong></p>';
        echo '<p>Default pages have been created for you. You can now:</p>';
        echo '<ul style="margin-left: 20px;">';
        echo '<li>• Customize your site using <a href="' . admin_url('customize.php') . '">Appearance → Customize</a></li>';
        echo '<li>• Install WooCommerce for e-commerce functionality</li>';
        echo '<li>• Edit pages content in <a href="' . admin_url('edit.php?post_type=page') . '">Pages</a></li>';
        echo '<li>• Add your products and configure your store</li>';
        echo '</ul>';
        echo '</div>';
    }
    
    // Check if pages exist, if not show creation notice
    if ($pagenow == 'edit.php' && isset($_GET['post_type']) && $_GET['post_type'] == 'page') {
        $existing_pages = get_posts(array(
            'post_type' => 'page',
            'meta_key' => '_fashion_boutique_default_page',
            'meta_value' => 'yes',
            'posts_per_page' => 1
        ));
        
        if (empty($existing_pages)) {
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p><strong>Missing Default Pages!</strong></p>';
            echo '<p>It looks like the default pages were not created. <a href="' . wp_nonce_url(admin_url('admin-post.php?action=create_fashion_pages'), 'create_pages') . '" class="button">Create Default Pages Now</a></p>';
            echo '</div>';
        }
    }
}
add_action('admin_notices', 'fashion_boutique_admin_notice');

/**
 * Handle manual page creation
 */
function fashion_boutique_handle_create_pages() {
    if (!current_user_can('manage_options')) {
        wp_die('You do not have permission to perform this action.');
    }
    
    if (!wp_verify_nonce($_GET['_wpnonce'], 'create_pages')) {
        wp_die('Security check failed.');
    }
    
    fashion_boutique_create_default_pages();
    
    wp_redirect(admin_url('edit.php?post_type=page&created=1'));
    exit;
}
add_action('admin_post_create_fashion_pages', 'fashion_boutique_handle_create_pages');

/**
 * Clean up default pages on theme deactivation (optional)
 */
function fashion_boutique_deactivation_cleanup() {
    // Only uncomment if you want to remove default pages when theme is deactivated
    /*
    $pages = get_posts(array(
        'post_type' => 'page',
        'meta_key' => '_fashion_boutique_default_page',
        'meta_value' => 'yes',
        'posts_per_page' => -1
    ));
    
    foreach ($pages as $page) {
        wp_delete_post($page->ID, true);
    }
    */
}
// add_action('switch_theme', 'fashion_boutique_deactivation_cleanup');
?>